<?php
if (!defined('ABSPATH')) exit;

/**
 * Handles the AJAX request for generating a preview.
 * Default values are applied ONLY when fetch is false (manual mode).
 */
function ehm_ajax_preview() {
    // 1. Permission Check
    if (!current_user_can('manage_options')) {
        wp_send_json_error('No Permission');
    }

    // 2. Get RAW data from POST.
    $url_raw      = isset($_POST['url']) ? esc_url_raw(trim($_POST['url'])) : '';
    $template     = isset($_POST['template']) ? max(1, intval($_POST['template'])) : 1;
    $color        = isset($_POST['color']) ? max(1, intval($_POST['color'])) : 1;
    $fetch        = !empty($_POST['fetch']) && $_POST['fetch'] === 'true';
    $title_raw    = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : '';
    $image_raw    = isset($_POST['image']) ? esc_url_raw($_POST['image']) : '';
    $cta_text_raw = isset($_POST['cta_text']) ? sanitize_text_field($_POST['cta_text']) : '';
    $icon_raw     = isset($_POST['icon']) ? sanitize_text_field($_POST['icon']) : '';
    
    $desc_raw = '';
    if (isset($_POST['description'])) {
        $desc_raw = sanitize_textarea_field($_POST['description']);
    } elseif (isset($_POST['desc'])) {
        $desc_raw = sanitize_textarea_field($_POST['desc']);
    }

    $icon_position  = isset($_POST['icon_position']) && in_array($_POST['icon_position'], array('left','right')) ? $_POST['icon_position'] : 'right';

    // 3. Color processing
    $gradient_colors = [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,23,24,25,26,27,28,29,30];
    $is_gradient = in_array($color, $gradient_colors);
    $color_type = $is_gradient ? 'gradient' : 'solid';

    $data = array();

    // 4. Logic based on 'fetch' mode
    if ($fetch && !empty($url_raw)) {
        // --- FETCH MODE ---
        require_once ehm_PATH . 'includes/class-json-fetch.php';
        $fetched = ehm_fetch_data($url_raw);

        if (isset($fetched['error'])) {
            wp_send_json_error($fetched['error']);
        }
        
        $data = $fetched;

        // Override with manual inputs if provided
        if (!empty($title_raw))    $data['title'] = $title_raw;
        if (!empty($desc_raw))     $data['description'] = $desc_raw;
        if (!empty($image_raw))    $data['image'] = $image_raw;
        if (!empty($cta_text_raw)) $data['cta_text'] = $cta_text_raw;
        
        if (empty($data['permalink'])) {
            $data['permalink'] = $url_raw;
        }
        $data['source']        = $data['source'] ?? 'fetch';
        $data['color']         = $color;
        $data['icon']          = $icon_raw;
        $data['icon_position'] = $icon_position;
        $data['gradient']      = $is_gradient;
        $data['color_type']    = $color_type;

    } else {
        // --- MANUAL MODE (fetch is false) ---
        // Apply default values for empty fields.

        $title    = !empty($title_raw) ? $title_raw : 'عنوان را وارد کنید';
        $desc     = !empty($desc_raw) ? $desc_raw : 'توضیحات را وارد کنید';
        $cta_text = !empty($cta_text_raw) ? $cta_text_raw : 'متن لینک شما';
        $url      = !empty($url_raw) ? $url_raw : '#';

        // === START: کد جدید برای تصویر پیش‌فرض ===
        $image = $image_raw;
        if (empty($image_raw)) {
            // ehm_PATH معمولاً مسیر پوشه پلاگین است.
            // پس یک دایرکتوری به عقب برمی‌گردیم و وارد assets/img می‌شویم.
            // plugins_url() مسیر URL معتبر برای فایل را تولید می‌کند.
            // فرض می‌شود که ehm_PATH مسیر ریشه پلاگین شما را برمی‌گرداند.
            // اگر ehm_PATH مسیر یک زیرپوشه است (مثلا /includes/)، باید مسیر را متناسب با آن تنظیم کنید.
            // مثال زیر فرض می‌کند ehm_PATH به ریشه پلاگین اشاره دارد.
            // اگر ehm_PATH به '.../plugin-name/includes/' اشاره دارد، باید از '../assets/img/no-image.png' استفاده کنید.
            $image = plugins_url('../assets/img/no-image.png', __FILE__);
        }
        // === END: کد جدید برای تصویر پیش‌فرض ===

        $data = array(
            'title'         => $title,
            'description'   => $desc,
            'image'         => $image, // استفاده از متغیر image که مقدار پیش‌فرض را دارد
            'type'          => '',
            'price'         => '',
            'permalink'     => $url,
            'source'        => 'manual',
            'color'         => $color,
            'cta_text'      => $cta_text,
            'icon'          => $icon_raw,
            'icon_position' => $icon_position,
            'gradient'      => $is_gradient,
            'color_type'    => $color_type,
        );
    }

    // 5. Ensure all required keys exist to prevent errors in templates
    $required_keys = array('title', 'description', 'image', 'type', 'price', 'permalink', 'source', 'color', 'cta_text', 'icon', 'icon_position', 'gradient', 'color_type');
    foreach ($required_keys as $key) {
        if (!array_key_exists($key, $data)) {
            $data[$key] = '';
        }
    }

    // 6. Prepare attributes for the template file
    $atts = array(
        'template'      => $template,
        'color'         => $color,
        'cta_text'      => $data['cta_text'],
        'icon'          => $data['icon'],
        'icon_position' => $icon_position,
        'gradient'      => $is_gradient,
        'color_type'    => $color_type,
    );

    // 7. Load the template and generate HTML
    $template_file = ehm_PATH . "includes/templates/template{$template}.php";

    if (file_exists($template_file)) {
        ob_start();
        include $template_file;
        $html = ob_get_clean();

        $response = array(
            'html' => $html,
            'meta' => $data
        );
        wp_send_json_success($response);

    } else {
        wp_send_json_error('Template not found');
    }
}

add_action( 'wp_ajax_ehm_preview', 'ehm_ajax_preview' );
