<?php
if ( ! defined( 'ABSPATH' ) ) exit; 

function ehm_fetch_data($url) {
    $url  = esc_url_raw($url);
    $home = home_url();

    if (strpos($url, $home) === 0) {
        $path  = trim(parse_url($url, PHP_URL_PATH), '/');
        $parts = explode('/', $path);
        $slug  = end($parts);

        $product = get_page_by_path($slug, OBJECT, 'product');
        if ($product) {
            $wc_product = wc_get_product($product->ID);
            if ($wc_product) {
                return [
                    'type'           => 'product',
                    'id'             => $product->ID,
                    'title'          => $wc_product->get_name(),
                    'description'    => wp_strip_all_tags($wc_product->get_short_description()),
                    'image'          => wp_get_attachment_url($wc_product->get_image_id()),
                    'price'          => $wc_product->get_price(),            
                    'regular_price'  => $wc_product->get_regular_price(),   
                    'sale_price'     => $wc_product->get_sale_price(),      
                    'permalink'      => $wc_product->get_permalink(),
                    'source'         => 'wc-local'
                ];
            }
        }

        $post = get_page_by_path($slug, OBJECT, 'post');
        if ($post) {
            return [
                'type'        => 'post',
                'id'          => $post->ID,
                'title'       => get_the_title($post),
                'description' => wp_trim_words(strip_tags($post->post_content), 30),
                'image'       => get_the_post_thumbnail_url($post->ID, 'full'),
                'permalink'   => get_permalink($post->ID),
                'source'      => 'wp-local'
            ];
        }

        $page = get_page_by_path($slug, OBJECT, 'page');
        if ($page) {
            return [
                'type'        => 'page',
                'id'          => $page->ID,
                'title'       => get_the_title($page),
                'description' => wp_trim_words(strip_tags($page->post_content), 30),
                'image'       => get_the_post_thumbnail_url($page->ID, 'full'),
                'permalink'   => get_permalink($page->ID),
                'source'      => 'wp-local'
            ];
        }

        $term = get_term_by('slug', $slug, 'product_cat');
        if (!$term) {
            $term = get_term_by('slug', $slug, 'category');
        }
        if ($term) {
            $image_url = '';

            if (function_exists('get_term_meta')) {
                $thumb_id = get_term_meta($term->term_id, 'thumbnail_id', true);
                if ($thumb_id) {
                    $image_url = wp_get_attachment_url($thumb_id);
                }
            }

            $description = $term->description;

            if (empty($image_url) || empty($description)) {
                $category_url = get_term_link($term->term_id);
                $response     = wp_remote_get($category_url);

                if (!is_wp_error($response)) {
                    $html = wp_remote_retrieve_body($response);
                    $schema_data = ehm_parse_schema_image_and_description($html);

                    if (empty($image_url) && !empty($schema_data['image'])) {
                        $image_url = $schema_data['image'];
                    }
                    if (empty($description) && !empty($schema_data['description'])) {
                        $description = $schema_data['description'];
                    }
                }
            }

            return [
                'type'        => 'category',
                'id'          => $term->term_id,
                'title'       => $term->name,
                'description' => $description,
                'image'       => $image_url,
                'permalink'   => get_term_link($term->term_id),
                'source'      => 'wp-local'
            ];
        }

        return ['error' => 'شناسه محتوای داخلی پیدا نشد'];
    }

    return ehm_fetch_external_data($url);
}


function ehm_parse_schema_image_and_description($html) {
    $image_url = '';
    $description_txt = '';

    if (empty($html)) return ['image' => '', 'description' => ''];

    libxml_use_internal_errors(true);
    $dom = new DOMDocument();
    $dom->loadHTML('<?xml encoding="utf-8" ?>' . $html);
    libxml_clear_errors();

    $scripts = $dom->getElementsByTagName('script');
    $schemas = [];

    foreach ($scripts as $script) {
        if (strtolower($script->getAttribute('type')) === 'application/ld+json') {
            $json_content = trim($script->nodeValue);
            $json_candidates = [];
            if (strpos($json_content, "\n") !== false && substr_count($json_content, '{') > 1) {
                preg_match_all('/\{.*?\}/s', $json_content, $all_jsons);
                foreach ($all_jsons[0] as $j) $json_candidates[] = $j;
            } else {
                $json_candidates[] = $json_content;
            }
            foreach ($json_candidates as $json_candidate) {
                $data = json_decode($json_candidate, true);
                if (json_last_error() !== JSON_ERROR_NONE || !$data) continue;
                if (isset($data['@graph']) && is_array($data['@graph'])) {
                    foreach ($data['@graph'] as $item) $schemas[] = $item;
                } else {
                    $schemas[] = $data;
                }
            }
        }
    }

    foreach ($schemas as $item) {
        if (
            isset($item['@type']) 
            && strtolower($item['@type']) === 'imageobject'
        ) {
            if (!empty($item['thumbnailUrl'])) {
                $image_url = esc_url_raw($item['thumbnailUrl']);
                break;
            } elseif (!empty($item['contentUrl'])) {
                $image_url = esc_url_raw($item['contentUrl']);
                break;
            } elseif (!empty($item['url'])) {
                $image_url = esc_url_raw($item['url']);
                break;
            }
        }
    }
    if (empty($image_url)) {
        foreach ($schemas as $item) {
            if (isset($item['image'])) {
                if (is_array($item['image']) && isset($item['image']['url'])) {
                    $image_url = esc_url_raw($item['image']['url']);
                } elseif (is_string($item['image'])) {
                    $image_url = esc_url_raw($item['image']);
                }
                if ($image_url) break;
            }
            if (isset($item['primaryImageOfPage']) && isset($item['primaryImageOfPage']['@id'])) {
                foreach ($schemas as $inner) {
                    if (isset($inner['@id']) && $inner['@id'] == $item['primaryImageOfPage']['@id'] && isset($inner['url'])) {
                        $image_url = esc_url_raw($inner['url']);
                        break 2;
                    }
                }
            }
        }
    }

    $page_url = '';
    foreach ($schemas as $item) {
        if (
            isset($item['@type']) 
            && in_array(strtolower($item['@type']), ['collectionpage','webpage']) 
            && isset($item['url']) 
            && filter_var($item['url'], FILTER_VALIDATE_URL)
        ) {
            $page_url = $item['url'];
            break;
        }
    }

    if ($page_url) {
        foreach ($schemas as $item) {
            if (
                isset($item['@type'])
                && in_array(strtolower($item['@type']), ['collectionpage','webpage'])
                && isset($item['description']) && !empty($item['description'])
                && isset($item['url']) && $item['url'] == $page_url
            ) {
                $description_txt = trim(strip_tags($item['description']));
                break;
            }
        }
    }
    if (empty($description_txt)) {
        foreach ($schemas as $item) {
            if (
                isset($item['@type']) 
                && in_array(strtolower($item['@type']), ['collectionpage','webpage'])
                && isset($item['description']) && !empty($item['description'])
            ) {
                $description_txt = trim(strip_tags($item['description']));
                break;
            }
        }
    }
    if (empty($description_txt)) {
        foreach ($schemas as $item) {
            if (
                isset($item['@type']) 
                && strtolower($item['@type']) === 'website'
                && isset($item['description']) && !empty($item['description'])
            ) {
                $description_txt = trim(strip_tags($item['description']));
                break;
            }
        }
    }
    if (empty($description_txt)) {
        foreach ($schemas as $item) {
            if (isset($item['description']) && !empty($item['description'])) {
                $description_txt = trim(strip_tags($item['description']));
                break;
            }
        }
    }

    return [
        'image'       => $image_url,
        'description' => $description_txt
    ];
}


function ehm_fetch_external_data($url) {
    $url    = esc_url_raw($url);
    $parsed = parse_url($url);
    if (empty($parsed['scheme']) || empty($parsed['host'])) {
        return ['error' => 'لینک خارجی نامعتبر است'];
    }
    $domain = $parsed['scheme'] . '://' . $parsed['host'];

    $response = wp_remote_get($url);
    if (is_wp_error($response)) {
        return ['error' => 'عدم دسترسی به لینک خارجی'];
    }
    $html = wp_remote_retrieve_body($response);

    libxml_use_internal_errors(true);
    $dom = new DOMDocument();
    $dom->loadHTML('<?xml encoding="utf-8" ?>' . $html);
    libxml_clear_errors();

    $body_classes_str = '';
    $body_tags = $dom->getElementsByTagName('body');
    if ($body_tags->length > 0) {
        $body_classes_str = $body_tags->item(0)->getAttribute('class');
    }
    if (empty($body_classes_str)) {
        $schema_data = ehm_parse_schema_image_and_description($html);
        return [
            'type'        => 'unknown',
            'id'          => null,
            'title'       => '',
            'description' => $schema_data['description'],
            'image'       => $schema_data['image'],
            'price'       => '',
            'regular_price' => '',
            'sale_price'    => '',
            'permalink'   => $url,
            'source'      => 'wp-external'
        ];
    }

    $body_classes = preg_split('/\s+/', trim($body_classes_str));
    $post_id      = null;
    $content_type = null;

    foreach ($body_classes as $class) {
        if (strpos($class, 'postid-') === 0) {
            $post_id = (int) str_replace('postid-', '', $class);
        }
        if ($class === 'single-product') {
            $content_type = 'product';
        }
        if ($class === 'single-post') {
            $content_type = 'post';
        }
    }
    if (!$post_id || !$content_type) {
        $schema_data = ehm_parse_schema_image_and_description($html);
        return [
            'type'        => 'unknown',
            'id'          => null,
            'title'       => '',
            'description' => $schema_data['description'],
            'image'       => $schema_data['image'],
            'price'       => '',
            'regular_price' => '',
            'sale_price'    => '',
            'permalink'   => $url,
            'source'      => 'wp-external'
        ];
    }

    if ($content_type === 'product') {
        $api_url = $domain . '/wp-json/wp/v2/product/' . $post_id;
    } else {
        $api_url = $domain . '/wp-json/wp/v2/posts/' . $post_id;
    }

    $api_response = wp_remote_get($api_url);
    $api_data = [];
    if (!is_wp_error($api_response)) {
        $api_data = json_decode(wp_remote_retrieve_body($api_response), true);
    }

    $description_txt = '';
    if (!empty($api_data['excerpt']['rendered'])) {
        $description_txt = wp_strip_all_tags($api_data['excerpt']['rendered']);
    }

    $image_url = '';
    if (!empty($api_data['_links']['wp:featuredmedia'][0]['href'])) {
        $media_url = $api_data['_links']['wp:featuredmedia'][0]['href'];
        $media_response = wp_remote_get($media_url);
        if (!is_wp_error($media_response)) {
            $media_data = json_decode(wp_remote_retrieve_body($media_response), true);
            if (json_last_error() === JSON_ERROR_NONE && !empty($media_data['guid']['rendered'])) {
                $image_url = esc_url_raw($media_data['guid']['rendered']);
            }
        }
    }

    $regular_price = isset($api_data['regular_price']) ? $api_data['regular_price'] : '';
    $sale_price    = isset($api_data['sale_price'])    ? $api_data['sale_price']    : '';
    $price         = !empty($sale_price) ? $sale_price : $regular_price;

    if ($content_type === 'product' && empty($price)) {
        if (preg_match('/<span[^>]*class="woocommerce-Price-amount[^"]*"[^>]*>(.*?)<\/span>/', $html, $matches)) {
            $html_price = trim(strip_tags($matches[1]));
            $price = $regular_price = $html_price;
        }
        if (preg_match('/<ins[^>]*>.*?<span[^>]*class="woocommerce-Price-amount[^"]*"[^>]*>(.*?)<\/span>.*?<\/ins>/s', $html, $ins_match)) {
            $sale_price = trim(strip_tags($ins_match[1]));
            $price = $sale_price;
        }
        if (preg_match('/<del[^>]*>.*?<span[^>]*class="woocommerce-Price-amount[^"]*"[^>]*>(.*?)<\/span>.*?<\/del>/s', $html, $del_match)) {
            $regular_price = trim(strip_tags($del_match[1]));
        }
    }

    $schema_data = ehm_parse_schema_image_and_description($html);
    if (empty($image_url))        $image_url = $schema_data['image'];
    if (empty($description_txt))  $description_txt = $schema_data['description'];

    return [
        'type'           => $content_type,
        'id'             => $post_id,
        'title'          => isset($api_data['title']['rendered']) ? wp_strip_all_tags($api_data['title']['rendered']) : '',
        'description'    => $description_txt,
        'image'          => $image_url,
        'price'          => $price,
        'regular_price'  => $regular_price,
        'sale_price'     => $sale_price,
        'permalink'      => $url,
        'source'         => 'wp-external'
    ];
}

if (!function_exists('is_assoc')) {
    function is_assoc(array $arr) {
        if ($arr === []) return false;
        return array_keys($arr) !== range(0, count($arr) - 1);
    }
}
